using System;
using System.Collections.Generic;
using System.Text;
using Server;
using Server.Engines.Lua;
using RemoteScripting;
using LuaInterface;

namespace Server.Items
{
	class LuaItem : Item, ILuaCodeConsumer
	{
		private Lua m_Lua;
		private LuaCode m_Code;
		private string m_Script;

		public LuaInterface.Lua Lua
		{
			get 
            {
                if (m_Lua == null)
                    m_Lua = new LuaInterface.Lua();
                return m_Lua; 
            }
		}

		public LuaCode Code
		{
			get { return m_Code; }
			set
			{
				if ( m_Code != null )
				{
					m_Code.Unsubscribe( this );
				}

				m_Code = value;

				if ( m_Code != null )
				{
					m_Code.Subscribe( this );
				}
			}
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public string Script
		{
			get { return m_Script; }
			set
			{
				m_Script = value;

				Code = (LuaCode)LuaCodeBase.Scripts[m_Script];
			}
		}

		public string Description
		{
			get { return String.Format( "Item '{0}', Type '{1}', Serial {2}", this.Name, this.GetType().ToString(), this.Serial.Value.ToString() ); }
		}

		private void Update()
		{
			if ( m_Code != null )
			{
				m_Lua = new LuaInterface.Lua();

				try
				{
					m_Lua["this"] = this;

					m_Lua.DoString( m_Code.Code );

					LuaHelper.CallFunction( this, "OnInitialize" );
				}
				catch ( Exception e )
				{
					LuaHelper.BroudcastError( this, e.Message );
				}
			}
		}

		public void OnLostSubscription()
		{
			Script = null;
			Update();
		}

		public void OnScriptChanged()
		{
			Update();
		}

		public override string DefaultName { get { return "Lua Scripted Item"; } }

		[Constructable]
		public LuaItem()
			: base( 0xE77 )
		{
		}

		public override void OnAdded( object parent )
		{
			LuaHelper.CallFunction( this, "OnAdded", parent );
		}

		public override void OnRemoved( object parent )
		{
			LuaHelper.CallFunction( this, "OnRemoved", parent );
		}

		public override bool OnDecay()
		{
			return LuaHelper.CallFunction( this, "OnDecay" ).AsBoolean( 0, base.OnDecay() );
		}

		public override void OnDoubleClick( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnDoubleClick", from );
		}

		public override bool OnDragDrop( Mobile from, Item dropped )
		{
			return LuaHelper.CallFunction( this, "OnDragDrop", from, dropped ).AsBoolean( 0, base.OnDragDrop( from, dropped ) );
		}

		public override bool OnDragLift( Mobile from )
		{
			return LuaHelper.CallFunction( this, "OnDragLift", from ).AsBoolean( 0, base.OnDragLift( from ) );
		}

		public override bool OnDroppedInto( Mobile from, Server.Items.Container target, Point3D p )
		{
			return LuaHelper.CallFunction( this, "OnDroppedInto", from, target, p ).AsBoolean( 0, base.OnDroppedInto( from, target, p ) );
		}

		public override bool OnDroppedOnto( Mobile from, Item target )
		{
			return LuaHelper.CallFunction( this, "OnDroppedOnto", from, target ).AsBoolean( 0, base.OnDroppedOnto( from, target ) );
		}

		public override bool OnDroppedToMobile( Mobile from, Mobile target )
		{
			return LuaHelper.CallFunction( this, "OnDroppedToMobile", from, target ).AsBoolean( 0, base.OnDroppedToMobile( from, target ) );
		}

		public override bool OnDroppedToWorld( Mobile from, Point3D p )
		{
			return LuaHelper.CallFunction( this, "OnDroppedToWorld", from, p ).AsBoolean( 0, base.OnDroppedToWorld( from, p ) );
		}

		public override bool OnEquip( Mobile from )
		{
			return LuaHelper.CallFunction( this, "OnEquip", from ).AsBoolean( 0, base.OnEquip( from ) );
		}

		public override void OnItemUsed( Mobile from, Item item )
		{
			LuaHelper.CallFunction( this, "OnItemUsed", from, item );
		}

		public override void OnLocationChange( Point3D oldLocation )
		{
			LuaHelper.CallFunction( this, "OnLocationChange", oldLocation );
		}

		public override void OnMapChange()
		{
			LuaHelper.CallFunction( this, "OnMapChange" );
		}

		public override void OnMovement( Mobile m, Point3D oldLocation )
		{
			LuaHelper.CallFunction( this, "OnMovement", m, oldLocation );
		}

		public override bool OnMoveOff( Mobile m )
		{
			return LuaHelper.CallFunction( this, "OnMoveOff", m ).AsBoolean( 0, base.OnMoveOff( m ) );
		}

		public override bool OnMoveOver( Mobile m )
		{
			return LuaHelper.CallFunction( this, "OnMoveOver", m ).AsBoolean( 0, base.OnMoveOff( m ) );
		}

		public override DeathMoveResult OnParentDeath( Mobile parent )
		{
			return (DeathMoveResult)LuaHelper.CallFunction( this, "OnParentDeath", parent ).AsInteger( 0, (int)base.OnParentDeath( parent ) );
		}

		public override void OnParentDeleted( object parent )
		{
			LuaHelper.CallFunction( this, "OnParentDeleted", parent );
		}

		public override void OnSecureTrade( Mobile from, Mobile to, Mobile newOwner, bool accepted )
		{
			LuaHelper.CallFunction( this, "OnSecureTrade", from, to, newOwner, accepted );
		}

		public override void OnSingleClick( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnSingleClick", from );
		}

		public override void OnSingleClickContained( Mobile from, Item item )
		{
			LuaHelper.CallFunction( this, "OnSingleClickContained", from, item );
		}

		public override void OnSnoop( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnSnoop", from );
		}

		public override void OnSpeech( SpeechEventArgs e )
		{
			LuaHelper.CallFunction( this, "OnSpeech", e );
		}

		public override void OnSubItemAdded( Item item )
		{
			LuaHelper.CallFunction( this, "OnSubItemAdded", item );
		}

		public override void OnSubItemRemoved( Item item )
		{
			LuaHelper.CallFunction( this, "OnSubItemRemoved", item );
		}

		public LuaItem( Serial serial )
			: base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int)1 ); // version	

			writer.Write( (string)m_Script );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
					{
						Script = reader.ReadString();
						break;
					}
			}
		}
	}
}